
/* 
 *
 * Stereo Audio Delay/Echo/Reverb/DSP code
 * (c) 2013 SILICON CHIP Publications
 * Author: Nicholas Vinen
 *
 * Please note that this code is not optimal, eg, SPI1 (audio in/out)
 * is polled because the maximum DMA transfer size is 64kB and the RAM
 * buffer is nearly 128kB. It could be handled in a "ping-pong" manner
 * or even interrupt driven but the chip is fast to poll it in the
 * inner loop along with SRAM transfers (if SRAM is present) so that's
 * what I've done. For more serious DSP use, DMA could be re-enabled;
 * the code is present, it's just disabled but it works as-is on chips
 * with 64kB RAM.
 *
 * Chip: PIC32MX450F256H-I/PT
 * Pin assignments:
 *
 *   D+/D-      : USB D+/D-
 *
 *   RB0-RB4    : PMP address lines
 *   RB5/VBUSON : USB VBUS on (active high)
 *   RB6        : PGEC / right channel delay PWM output
 *   RB7        : PGED / left channel delay PWM output / pull down for echo
 *   RB8/AN8    : VR1 (Delay1 adjustment)
 *   RB9-RB15   : PMP address lines
 *   RC12/OSC1  : MCLK from WM8731
 *   RC13/RPC13 : serial audio data to WM8731
 *   RC14/RPC14 : AUX1 (expansion header pin 7)
 *   RC15       : N/C
 *
 *   RD0/RPD0   : MOSI for WM8731 control SPI
 *   RD1/AN24   : VR2 (Delay2/other adjustment)
 *   RD2/RPD2   : serial audio bit clock from WM8731
 *   RD3/RPD3   : serial audio data from WM8731
 *   RD4/PMWR   : SRAM write enable-bar
 *   RD5/PMRD   : SRAM output enable-bar
 *   RD6        : SRAM CS-bar
 *   RD7        : AUX4 (expansion header pin 1)
 *   RD8/RPD8   : N/C
 *   RD9/RPD9   : serial audio sample clock from WM8731
 *   RD10-RD11  : PMP address lines
 *
 *   RE0-RE7    : PMP data lines
 *
 *   RF0/RPF0   : CS-bar for WM8731 control SPI
 *   RF1/RPF1   : SCK for WM8731 control SPI
 *   RF3/USBID  : USBID
 *   RF4-RF5    : PMP address lines
 *
 *   RG6-RG9    : PMP address lines
 */

#include <plib.h>
#include <xc.h>
#include "WM8731.h"
#include "SRAM.h"

// DEVCFG3
// USERID = No Setting
#pragma config FSRSSEL = PRIORITY_7     // Shadow Register Set Priority Select (SRS Priority 7)
#pragma config PMDL1WAY = ON            // Peripheral Module Disable Configuration (Allow only one reconfiguration)
#pragma config IOL1WAY = ON             // Peripheral Pin Select Configuration (Allow only one reconfiguration)
#pragma config FUSBIDIO = ON            // USB USID Selection (Controlled by the USB Module)
#pragma config FVBUSONIO = ON           // USB VBUS ON Selection (Controlled by USB Module)

// DEVCFG2
#pragma config FPLLIDIV = DIV_3//DIV_2  // PLL Input Divider (2x Divider for FRC, 3x Divider for CLKI)
#pragma config FPLLMUL = MUL_20         // PLL Multiplier (20x Multiplier)
#pragma config UPLLIDIV = DIV_12        // USB PLL Input Divider (12x Divider)
#pragma config UPLLEN = OFF             // USB PLL Enable (Disabled and Bypassed)
#pragma config FPLLODIV = DIV_1         // System PLL Output Clock Divider (PLL Divide by 1)

// DEVCFG1
#pragma config FNOSC = PRIPLL//FRCPLL   // Oscillator Selection Bits (External clock with PLL or Fast RC Osc with PLL)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disabled)
#pragma config IESO = ON                // Internal/External Switch Over (Enabled)
#pragma config POSCMOD = EC             // Primary Oscillator Configuration (External clock/Primary osc disabled)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FPBDIV = DIV_1           // Peripheral Clock Divisor (Pb_Clk is Sys_Clk/1)
#pragma config FCKSM = CSDCMD           // Clock Switching and Monitor Selection (Clock Switch Disable, FSCM Disabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WINDIS = OFF             // Watchdog Timer Window Enable (Watchdog Timer is in Non-Window Mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled (SWDTEN Bit Controls))
#pragma config FWDTWINSZ = WISZ_25      // Watchdog Timer Window Size (Window Size is 25%)

// DEVCFG0
#pragma config DEBUG = ON               // Background Debugger Enable (Debugger is Enabled)
#pragma config JTAGEN = ON              // JTAG Enable (JTAG Port Enabled)
#pragma config ICESEL = ICS_PGx2        // ICE/ICD Comm Channel Select (Communicate on PGEC2/PGED2)
#pragma config PWP = OFF                // Program Flash Write Protect (Disable)
#pragma config BWP = OFF                // Boot Flash Write Protect bit (Protection Disabled)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)


#define SYS_FREQ (80000000L)

#define SRAM_SIZE             1048576
#define BUFFER_SIZE_WORDS    (65536-768)
#define SRAM_XFER_SIZE        64
#define ADJUST_MIN_DELTA      2
#define MIN_VAL_ZERO_DELAY    2
#define REVERB_SAMPLES        24
//#define REVERB_FRACTION       31 / 32

unsigned short DMA_Buffer[BUFFER_SIZE_WORDS];

unsigned int VirtToPhys(volatile void* p) {
  return (int)p<0?((int)p&0x1fffffffL):(unsigned int)((unsigned char*)p+0x40000000L);
}

void DMA0Setup(void) {
    // Set up DMA0 to continually stream data from SPI peripheral into DMA_Buffer starting at address 0
    // Once buffer is full, start again. Repeat forever.

    IEC2bits.DMA0IE = 0;           // disable DMA channel 0 interrupts
    IFS2bits.DMA0IF = 0;           // clear any existing DMA channel 0 interrupt flag
    DMACONbits.ON = 1;             // enable DMA unit
    DCH0CON = 0;                   // channel off, no chaining
    DCH0CONbits.CHPRI = 3;         // priority 3
    DCH0ECON = 0;
    DCH0ECONbits.SIRQEN = 1;       // start transfer on interrupt
    DCH0ECONbits.CHSIRQ = 36;      // SPI1RX interrupt
    DCH0SSA = VirtToPhys(&SPI1BUF);// transfer source physical address
    DCH0DSA = VirtToPhys(DMA_Buffer);// transfer destination physical address
    DCH0SSIZ = sizeof(*DMA_Buffer); // source size
    DCH0DSIZ = sizeof(DMA_Buffer); // destination size
    DCH0CSIZ = sizeof(*DMA_Buffer);// bytes transferred per event
    DCH0INTCLR = 0x00ff00ff;       // clear existing events, disable all interrupts
    DCH0CONbits.CHAEN = 1;         // auto-enable so that a new block transfer begins as soon as the last finishes
    DCH0CONbits.CHEN = 1;          // turn channel on
}

void DMA1Setup(void) {
    // Set up DMA1 to continually stream data to SPI peripheral from DMA_Buffer starting at address 0
    // Once buffer has been transferred, start again. Repeat forever.

    IEC2bits.DMA1IE = 0;           // disable DMA channel 1 interrupts
    IFS2bits.DMA1IF = 0;           // clear any existing DMA channel 1 interrupt flag
    DMACONbits.ON = 1;             // enable DMA unit
    DCH1CON = 0;                   // channel off, no chaining
    DCH1CONbits.CHPRI = 3;         // priority 3
    DCH1ECON = 0;
    DCH1ECONbits.SIRQEN = 1;       // start transfer on interrupt
    DCH1ECONbits.CHSIRQ = 37;      // SPI1RX interrupt
    DCH1SSA = VirtToPhys(DMA_Buffer);// transfer source physical address
    DCH1DSA = VirtToPhys(&SPI1BUF);// transfer destination physical address
    DCH1SSIZ = sizeof(DMA_Buffer); // source size
    DCH1DSIZ = sizeof(*DMA_Buffer);// destination size
    DCH1CSIZ = sizeof(*DMA_Buffer);// bytes transferred per event
    DCH1INTCLR = 0x00ff00ff;       // clear existing events, disable all interrupts
    DCH1CONbits.CHAEN = 1;         // auto-enable so that a new block transfer begins as soon as the last finishes
}

void ADCSetup(void) {
    AD1CHSbits.CH0SA = 8;   // input A = AN8
//    AD1CHSbits.CH0SB = 24;  // input B = AN24
    AD1CHS |= 24<<24;       // compensate for buggy header file
    AD1CON1bits.ASAM = 1;   // automatic sampling start
    AD1CON1bits.SSRC = 7;   // auto conversion
    AD1CON1bits.FORM = 0;   // unsigned 16-bit integer result
//    AD1CON2bits.ALTS = 1;   // alternate between inputs A & B
    AD1CON2bits.SMPI = 2;//1;   // interrupt after every three conversions
    AD1CON2bits.CSCNA = 1;  // scan inputs
    AD1CON3bits.ADCS = 255; // maximum TAD = 80MHz / 256 = 3.2us
    AD1CON3bits.SAMC = 31;  // 31 TAD sampling time
    AD1CSSLbits.CSSL5 = 1;
    AD1CSSLbits.CSSL8 = 1;
    AD1CSSLbits.CSSL24 = 1;
    IFS0bits.AD1IF = 0;     // set up interrupt
    IPC5bits.AD1IP = 1;
    IEC0bits.AD1IE = 1;
    AD1CON1bits.ON = 1;     // turn on ADC unit
    AD1CON1bits.SAMP = 1;   // start sampling
}

volatile unsigned short AN5Val, AN8Val, AN24Val;
volatile unsigned long AN5_Accum, AN8_Accum, AN24_Accum, ADCSubsampleCounter, ADCSampleCounter;
void __ISR(_ADC_VECTOR,IPL1) __ADCInterrupt(void) {
    AN5_Accum += ADC1BUF0;
    AN8_Accum += ADC1BUF1;
    AN24_Accum += ADC1BUF2;
    if( ++ADCSubsampleCounter == 8 ) {
        AN5Val = AN5_Accum / 8;
        AN8Val = AN8_Accum / 8;
        AN24Val = AN24_Accum / 8;
        AN5_Accum = 0;
        AN8_Accum = 0;
        AN24_Accum = 0;
        ADCSubsampleCounter = 0;
        ++ADCSampleCounter;
    }
    IFS0bits.AD1IF = 0;
}

void __ISR(_DMA1_VECTOR,IPL5) __DMA1Interrupt(void) {
    DCH1INTbits.CHSDIE = 0;
    DCH1INTbits.CHSDIF = 0;
    IEC2bits.DMA1IE = 0;
    IFS2bits.DMA1IF = 0;
    DCH1SSA = VirtToPhys(DMA_Buffer);// transfer source physical address
    DCH1SSIZ = sizeof(DMA_Buffer); // source size
    DCH1CONbits.CHAEN = 1;   // turn auto-enable back on
    DCH1CONbits.CHEN = 1;    // resume transmission
}

unsigned char SRAM_installed, dual_mono_mode, echo_mode, reverb_mode;
unsigned long internal_delay_bytes, external_delay_bytes, delay_pct, SRAM_readpos, SRAM_writepos;
unsigned long internal_delay2_bytes, external_delay2_bytes, delay2_pct, SRAM_readpos2, SRAM_writepos2;

#ifndef _DEBUG
static void update_pwm_outputs() {
    unsigned long delay_ms, period, delay2_ms, period2;
    if( dual_mono_mode ) {
        delay_ms = (internal_delay_bytes + external_delay_bytes) / (48 * 2);
        if( delay_ms )
            delay_ms += 2;
        period = delay_ms ? SYS_FREQ / delay_ms : 0;
        delay2_ms = (internal_delay2_bytes + external_delay2_bytes) / (48 * 2);
        if( delay2_ms )
            delay2_ms += 2;
        if( delay_ms < 5 ) {
            delay_ms = 5;
            period = SYS_FREQ / 5;
        }
        if( delay2_ms < 5 )
            delay2_ms = 5;
        period2 = delay2_ms ? SYS_FREQ / delay2_ms : 0;
    } else {
        delay_ms = (internal_delay_bytes + external_delay_bytes) / (48 * 4);
        if( delay_ms )
            delay_ms += 2;
        period = delay_ms ? SYS_FREQ / delay_ms : 0;
        delay2_ms = delay_ms;
        period2 = period;
    }

    OC1CON = 0;
    OC5CON = 0;
    T2CON = 0;
    T3CON = 0;

    if( !echo_mode || reverb_mode ) {
        // pin 4 of ICSP = PGED = RB7/AN7
        PORTSetPinsDigitalOut(IOPORT_B, BIT_7);
        LATBbits.LATB7 = 0;
        ANSELBbits.ANSB7 = 0;
        PPSOutput(3,RPB7,OC5);
    }
    if( !reverb_mode ) {
        // pin 5 of ICSP = PGEC = RB6/AN6
        PORTSetPinsDigitalOut(IOPORT_B, BIT_6);
        LATBbits.LATB6 = 0;
        ANSELBbits.ANSB6 = 0;
        PPSOutput(4,RPB6,OC1);
    }

    if( delay_pct > 0 || delay2_pct > 0 ) {
        unsigned long temp;
        TMR2 = 0;
        TMR3 = 0;
        if( dual_mono_mode ) {
            PR2 = period / 256;
            temp = period / 256 * delay_pct / 1024;
            if( PR2 && temp ) {
                OC5R = temp;
                OC5RS = temp;
                T2CONbits.TCKPS = 7; // divide by 256
                T2CONbits.ON = 1;
                OC5CONbits.OCM = 6; // PWM, fault pin disabled
                OC5CONbits.ON = 1;
            }
            PR3 = period2 / 256;
            temp = period2 / 256 * delay2_pct / 1024;
            if( PR3 && temp ) {
                OC1R = temp;
                OC1RS = temp;
                T3CONbits.TCKPS = 7; // divide by 256
                T3CONbits.ON = 1;
                OC1CONbits.OCM = 6; // PWM, fault pin disabled
                OC1CONbits.OCTSEL = 1;
                OC1CONbits.ON = 1;
            }
        } else {
            T2CONbits.T32 = 1;
            PR2 = period;//&65535;
            //PR3 = period>>16;
            temp = period * delay_pct / 1024;
            OC1R = temp;
            OC1RS = temp;
            OC5R = temp;
            OC5RS = temp;
            T2CONbits.ON = 1;
            OC1CONbits.OCM = 6; // PWM, fault pin disabled
            OC1CONbits.OC32 = 1;
            OC1CONbits.ON = 1;
            OC5CONbits.OCM = 6; // PWM, fault pin disabled
            OC5CONbits.OC32 = 1;
            OC5CONbits.ON = 1;
        }
    }
}
#endif

// stereo mode
void compute_delay(unsigned long ADCval) {
    if( SRAM_installed ) {
        internal_delay_bytes = ((SRAM_SIZE + sizeof(DMA_Buffer) - 1024) * ADCval / 1024) & ~3;
        if( internal_delay_bytes > sizeof(DMA_Buffer) - 512 ) {
            external_delay_bytes = (internal_delay_bytes - (sizeof(DMA_Buffer) - 512)) & ~(SRAM_XFER_SIZE-1);
            internal_delay_bytes -= external_delay_bytes;
            SRAM_readpos = SRAM_writepos + SRAM_SIZE - external_delay_bytes;
            if( SRAM_readpos > SRAM_SIZE )
                SRAM_readpos -= SRAM_SIZE;
        } else {
            external_delay_bytes = 0;
        }
    } else {
        internal_delay_bytes = (sizeof(DMA_Buffer) * ADCval / 1024) & ~3;
    }
    delay_pct = ADCval;
#ifndef _DEBUG
    update_pwm_outputs();
#endif
}

// dual mono mode
void compute_delays(unsigned long ADCval, unsigned long ADCval2) {
    if( SRAM_installed ) {
        internal_delay_bytes = ((SRAM_SIZE + sizeof(DMA_Buffer) - 1024) / 2 * ADCval / 1024) & ~3;
        if( internal_delay_bytes < 16 )
            internal_delay_bytes = 16;
        if( internal_delay_bytes > sizeof(DMA_Buffer) / 2 - 512 ) {
            external_delay_bytes = (internal_delay_bytes - (sizeof(DMA_Buffer) / 2 - 512)) & ~(SRAM_XFER_SIZE-1);
            internal_delay_bytes -= external_delay_bytes;
            SRAM_readpos = SRAM_writepos + SRAM_SIZE / 2 - external_delay_bytes;
            if( SRAM_readpos > SRAM_SIZE / 2 )
                SRAM_readpos -= SRAM_SIZE / 2;
        } else {
            external_delay_bytes = 0;
        }

        internal_delay2_bytes = ((SRAM_SIZE + sizeof(DMA_Buffer) - 1024) / 2 * ADCval2 / 1024) & ~3;
        if( internal_delay2_bytes < 16 )
            internal_delay2_bytes = 16;
        if( internal_delay2_bytes > sizeof(DMA_Buffer) / 2 - 512 ) {
            external_delay2_bytes = (internal_delay2_bytes - (sizeof(DMA_Buffer) / 2 - 512)) & ~(SRAM_XFER_SIZE-1);
            internal_delay2_bytes -= external_delay2_bytes;
            SRAM_readpos2 = SRAM_writepos2 + SRAM_SIZE / 2 - external_delay2_bytes;
            if( SRAM_readpos2 > SRAM_SIZE / 2 )
                SRAM_readpos2 -= SRAM_SIZE / 2;
        } else {
            external_delay2_bytes = 0;
        }
    } else {
        internal_delay_bytes = (sizeof(DMA_Buffer) / 2 * ADCval / 1024) & ~3;
        internal_delay2_bytes = (sizeof(DMA_Buffer) / 2 * ADCval2 / 1024) & ~3;
    }
    delay_pct = ADCval;
    delay2_pct = ADCval2;
#ifndef _DEBUG
    update_pwm_outputs();
#endif
}

unsigned char cur_hpvol;
void UpdateHeadphoneVol() {
    signed short new_hpvol = AN5Val * 80 / 1024 + 1;
    if( new_hpvol != cur_hpvol ) {
        WM8731_Set_Left_Headphone_Out_Vol(new_hpvol - 74, true, true);
        cur_hpvol = new_hpvol;
    }
}

unsigned char Mic_on, Bypass_on;
unsigned long MuteSamples;
void UpdateMicEnabled() {
    Mic_on = !PORTCbits.RC14;
    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
    MuteSamples = 24000; // mute for half a second after microphone input state changes
}

//    unsigned long temp, last_delay, last_delay2, new_delay, new_delay2;
//    unsigned char VR1_installed, VR2_installed, passthrough_mode = 0;
//    unsigned long copy_to_RAM_offset = 0, copy_to_RAM_offset2 = 0;
//    unsigned long write_offset, read_offset, write_offset2, read_offset2;
int main(void) {
    unsigned char VR1_installed, VR2_installed, VolumePot_installed, passthrough_mode;
    unsigned long temp, last_delay, last_delay2, new_delay, new_delay2;
#ifdef USE_DMA
    unsigned long current_DMA_write_offset;
#endif
    unsigned long copy_to_RAM_offset = 0, copy_to_RAM_offset2 = 0;
    unsigned long write_offset, read_offset, write_offset2, read_offset2;

    DDPCONbits.JTAGEN = 0;                            //  JTAG = OFF
    SYSTEMConfigPerformance(SYS_FREQ);
    INTEnableSystemMultiVectoredInt();
    INTEnableInterrupts();

    SRAMSetup();
    SRAM_installed = (TestSRAM((unsigned char*)DMA_Buffer, SRAM_SIZE) == 0);
    if( SRAM_installed )
        EraseSRAM((unsigned char*)DMA_Buffer, SRAM_SIZE);

    ADCSetup();
    temp = ADCSampleCounter;
    while( ADCSampleCounter == temp )
        ;
    CNPUBbits.CNPUB8 = 1;
    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;
    VR1_installed = AN8Val < 0x380;
    CNPUBbits.CNPUB8 = 0;
    if( !VR1_installed ) {
        CNPDBbits.CNPDB8 = 1;
        temp = ADCSampleCounter;
        while( ADCSampleCounter < temp+8+2 )
            ;
        VR1_installed = AN8Val > 0x080;
        CNPDBbits.CNPDB8 = 0;
    }
    temp = ADCSampleCounter;
    while( ADCSampleCounter == temp )
        ;
    CNPUDbits.CNPUD1 = 1;
    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;
    VR2_installed = AN24Val < 0x380;
    CNPUDbits.CNPUD1 = 0;
    if( !VR2_installed ) {
        CNPDDbits.CNPDD1 = 1;
        temp = ADCSampleCounter;
        while( ADCSampleCounter < temp+8+2 )
            ;
        VR2_installed = AN24Val > 0x080;
        CNPDDbits.CNPDD1 = 0;
    }

    // check to see whether we have a pot on AN5/RB5, for volume control
    temp = ADCSampleCounter;
    while( ADCSampleCounter == temp )
        ;
    CNPUBbits.CNPUB5 = 1;
    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;
    VolumePot_installed = AN5Val < 0x380;
    CNPUBbits.CNPUB5 = 0;
    if( !VolumePot_installed ) {
        CNPDBbits.CNPDB5 = 1;
        temp = ADCSampleCounter;
        while( ADCSampleCounter < temp+8+2 )
            ;
        VolumePot_installed = AN5Val > 0x080;
        CNPDBbits.CNPDB5 = 0;
    }

    // pin 4 of ICSP = PGED = RB7/AN7
    PORTSetPinsDigitalIn(IOPORT_B, BIT_7);
    ANSELBbits.ANSB7 = 0;
    CNPUBbits.CNPUB7 = 1;
    // pin 5 of ICSP = PGEC = RB6/AN6
    PORTSetPinsDigitalIn(IOPORT_B, BIT_6);
    ANSELBbits.ANSB6 = 0;
    CNPUBbits.CNPUB6 = 1;
    // pin 7 of CON7 = RC14 which indicates microphone presence
    PORTSetPinsDigitalIn(IOPORT_C, BIT_14);
    CNPUCbits.CNPUC14 = 1;

    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;

    reverb_mode = (PORTBbits.RB6 == 0);
    echo_mode = reverb_mode || (PORTBbits.RB7 == 0);
    dual_mono_mode = !echo_mode && VR1_installed && VR2_installed;
    CNPUBbits.CNPUB7 = 0;
    CNPUBbits.CNPUB6 = 0;

    WM8731_Setup_Control_SPI();
    WM8731_Enable_MCU_Interface(true);
#ifdef ANALOG_PASS_THROUGH
    WM8731_Power_Control(true, false, false, false, true, true, true, false);
    WM8731_Set_Analog_Audio_Path(false, false, Line_Input, true, false, false, 0);
    while(1)
        ;
#else
    WM8731_Set_Left_Line_In_Vol(0, false, true);
    if( VolumePot_installed )
        UpdateHeadphoneVol();
    Mic_on = !PORTCbits.RC14;
    Bypass_on = false;
    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
    WM8731_Set_Digital_Audio_Interface_Format(Left_Justified_MSB_First, 16, Right_When_LRC_High, false /* DAC left/right swap */, Master_Mode, false /* invert bclk */);
    WM8731_Sampling_Control(256, 48000, 48000, false /* mclk_divide_by_two */, false/*true*/ /* clkout_divide_by_two */);
    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
    WM8731_Set_Digital_Audio_Path(false /* adc HPF */, 0 /* de-emphasis */, false /* soft mute */, false /* store HPF DC offset */);

    /* bit clock = SCK1 (RD2), word clock = RPD9, data in = RPD3, data out = RF0 */
    PORTSetPinsDigitalIn(IOPORT_D, BIT_3);
    ANSELDbits.ANSD3 = 0;
    PPSInput(2,SDI1,RPD3);
    PORTSetPinsDigitalIn(IOPORT_D, BIT_9);
    PPSInput(3,SS1,RPD9);
    PORTSetPinsDigitalIn(IOPORT_D, BIT_2);
    ANSELDbits.ANSD2 = 0;
    PORTSetPinsDigitalOut(IOPORT_F, BIT_0);
    PPSOutput(2,RPF0,SDO1);

    SPI1CON = SPI1CON2 = 0;   // Reset settings
    SPI1STATbits.SPIROV = 1; // Clear overflow, if set
    SPI1STATbits.SPITUR = 1; // Clear underflow, if set
    SPI1STAT = 0;
    SPI1BUF;      // Read any data

    SPI1CON2bits.AUDEN = 1;
    SPI1CON2bits.SPISGNEXT = 1;
    SPI1CON2bits.IGNROV = 1;
    SPI1CON2bits.IGNTUR = 1;
    SPI1CON2bits.AUDMOD = 1; // left-justified
    SPI1CONbits.MODE16 = 1; // 16-bit data, 16-bit FIFO, 32-bit channel, 64-bit frame
    SPI1CONbits.SSEN = 1;
    SPI1CONbits.FRMPOL = 1;
    SPI1CONbits.CKP = 1;
    SPI1CONbits.ENHBUF = 1;
    SPI1CONbits.STXISEL = 3;
    SPI1CONbits.SRXISEL = 1;
#ifdef USE_DMA
    if( !dual_mono_mode && !echo_mode ) {
        DMA0Setup();
        DMA1Setup();
    }
#endif
    SPI1CONbits.ON = 1;

    if( dual_mono_mode ) {
//        while( 1 ) {
//            unsigned long i;
        last_delay = AN8Val;
        last_delay2 = AN24Val;
        compute_delays(last_delay, last_delay2);
//            for( i = 0; i < 1000; ++i )
//                PowerSaveIdle();
//        }
    } else {
        if( !VR1_installed )
            last_delay = AN24Val;
        else
            last_delay = AN8Val;
        compute_delay(last_delay);

#ifdef USE_DMA
        if( !echo_mode ) {
            if( internal_delay_bytes )
                while( DCH0DPTR != internal_delay_bytes )
                    ;
            DCH1CONbits.CHEN = 1; // start transmission
        }
#endif
    }

//  passthrough_mode = 0;

    if( dual_mono_mode ) {
        unsigned short* DMA_Buffer2 = DMA_Buffer + BUFFER_SIZE_WORDS / 2;
        unsigned long cur_read_channel = 0, cur_write_channel = 0;

        write_offset = write_offset2 = 0;
        read_offset = (sizeof(DMA_Buffer) / 2 - internal_delay_bytes) / 2;
        read_offset2 = (sizeof(DMA_Buffer) / 2 - internal_delay2_bytes) / 2;
        while(1) {
            if( last_delay < MIN_VAL_ZERO_DELAY && last_delay2 < MIN_VAL_ZERO_DELAY ) {
                if( !passthrough_mode ) {
                    Bypass_on = true;
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    passthrough_mode = 1;
                }
            } else {
                if( passthrough_mode ) {
                    Bypass_on = false;
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    passthrough_mode = 0;
                }
            }

            if( external_delay_bytes && SRAM_readpos != SRAM_writepos ) {
                if( write_offset*2 < copy_to_RAM_offset || write_offset*2 >= copy_to_RAM_offset + SRAM_XFER_SIZE ) {
                    RAMtoSRAMCopy((unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_writepos, SRAM_XFER_SIZE);
                    SRAM_writepos += SRAM_XFER_SIZE;
                    if( SRAM_writepos == SRAM_SIZE / 2 )
                        SRAM_writepos = 0;
                    SRAMtoRAMCopy(SRAM_readpos, (unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_XFER_SIZE);
                    SRAM_readpos += SRAM_XFER_SIZE;
                    if( SRAM_readpos == SRAM_SIZE / 2 )
                        SRAM_readpos = 0;
                    copy_to_RAM_offset += SRAM_XFER_SIZE;
                    if( copy_to_RAM_offset == sizeof(DMA_Buffer) / 2 )
                        copy_to_RAM_offset = 0;
                }
            }
            if( external_delay2_bytes && SRAM_readpos2 != SRAM_writepos2 ) {
                if( write_offset2*2 < copy_to_RAM_offset2 || write_offset2*2 >= copy_to_RAM_offset2 + SRAM_XFER_SIZE ) {
                    RAMtoSRAMCopy((unsigned char*)DMA_Buffer2 + copy_to_RAM_offset2, SRAM_writepos2 + SRAM_SIZE / 2, SRAM_XFER_SIZE);
                    SRAM_writepos2 += SRAM_XFER_SIZE;
                    if( SRAM_writepos2 == SRAM_SIZE / 2 )
                        SRAM_writepos2 = 0;
                    SRAMtoRAMCopy(SRAM_readpos2 + SRAM_SIZE / 2, (unsigned char*)DMA_Buffer2 + copy_to_RAM_offset2, SRAM_XFER_SIZE);
                    SRAM_readpos2 += SRAM_XFER_SIZE;
                    if( SRAM_readpos2 == SRAM_SIZE / 2 )
                        SRAM_readpos2 = 0;
                    copy_to_RAM_offset2 += SRAM_XFER_SIZE;
                    if( copy_to_RAM_offset2 == sizeof(DMA_Buffer) / 2 )
                        copy_to_RAM_offset2 = 0;
                }
            }
//            } else {
//                PowerSaveIdle();
            while( 1 ) {
                if( !SPI1STATbits.SPIRBE ) {
                    if( cur_read_channel ) {
                        signed short val = SPI1BUF;
                        if( MuteSamples ) {
                            val = 0;
                            --MuteSamples;
                        }
                        DMA_Buffer2[write_offset2] = val;
                        if( ++write_offset2 == BUFFER_SIZE_WORDS / 2 )
                            write_offset2 = 0;
                    } else {
                        signed short val = SPI1BUF;
                        if( MuteSamples ) {
                            val = 0;
                            --MuteSamples;
                        }
                        DMA_Buffer[write_offset] = val;
                        if( ++write_offset == BUFFER_SIZE_WORDS / 2 )
                            write_offset = 0;
                    }
                    cur_read_channel ^= 1;
                } else if( !SPI1STATbits.SPITBF ) {
                    if( cur_write_channel ) {
                        SPI1BUF = DMA_Buffer2[read_offset2];
                        if( ++read_offset2 == BUFFER_SIZE_WORDS / 2 )
                            read_offset2 = 0;
                    } else {
                        SPI1BUF = DMA_Buffer[read_offset];
                        if( ++read_offset == BUFFER_SIZE_WORDS / 2 )
                            read_offset = 0;
                    }
                    cur_write_channel ^= 1;
                } else {
                    break;
                }
            }

            new_delay = AN8Val;
            new_delay2 = AN24Val;
            if( (last_delay > 2 && new_delay <= last_delay - ADJUST_MIN_DELTA || new_delay >= last_delay + ADJUST_MIN_DELTA) ||
                (last_delay2 > 2 && new_delay2 <= last_delay2 - ADJUST_MIN_DELTA || new_delay2 >= last_delay2 + ADJUST_MIN_DELTA) ) {
                // delay changed
                compute_delays(new_delay, new_delay2);
                read_offset = write_offset + (sizeof(DMA_Buffer) / 2 - internal_delay_bytes) / 2;
                if( read_offset > sizeof(DMA_Buffer) / 4 )
                    read_offset -= sizeof(DMA_Buffer) / 4;
                read_offset2 = write_offset2 + (sizeof(DMA_Buffer) / 2 - internal_delay2_bytes) / 2;
                if( read_offset2 > sizeof(DMA_Buffer) / 4 )
                    read_offset2 -= sizeof(DMA_Buffer) / 4;
                last_delay = new_delay;
                last_delay2 = new_delay2;
            }
            if( VolumePot_installed )
                UpdateHeadphoneVol();
            if( !PORTCbits.RC14 != Mic_on )
                UpdateMicEnabled();
         }
    } else if( echo_mode ) {
        write_offset = 0;
        read_offset = (sizeof(DMA_Buffer) - internal_delay_bytes) / 2;
        if( reverb_mode )
            read_offset2 = (sizeof(DMA_Buffer) - REVERB_SAMPLES) / 2;
//            read_offset2 = (sizeof(DMA_Buffer) - ((internal_delay_bytes * REVERB_FRACTION) & ~3)) / 2;
        PORTSetPinsDigitalIn(IOPORT_D, BIT_7);
        CNPUDbits.CNPUD7 = 1;

        while(1) {
            unsigned long mix_val = VR1_installed && VR2_installed ? AN24Val : 768;

            if( last_delay < MIN_VAL_ZERO_DELAY ) {
                if( !passthrough_mode ) {
                    Bypass_on = true;
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    passthrough_mode = 1;
                }
            } else {
                if( passthrough_mode ) {
                    Bypass_on = false;
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    passthrough_mode = 0;
                }
            }

            if( external_delay_bytes ) {
                if( SRAM_readpos != SRAM_writepos ) {
                    if( write_offset < copy_to_RAM_offset || write_offset >= copy_to_RAM_offset + SRAM_XFER_SIZE ) {
                        RAMtoSRAMCopy((unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_writepos, SRAM_XFER_SIZE);
                        SRAM_writepos += SRAM_XFER_SIZE;
                        if( SRAM_writepos == SRAM_SIZE )
                            SRAM_writepos = 0;
                        SRAMtoRAMCopy(SRAM_readpos, (unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_XFER_SIZE);
                        SRAM_readpos += SRAM_XFER_SIZE;
                        if( SRAM_readpos == SRAM_SIZE )
                            SRAM_readpos = 0;
                        copy_to_RAM_offset += SRAM_XFER_SIZE;
                        if( copy_to_RAM_offset == sizeof(DMA_Buffer) )
                            copy_to_RAM_offset = 0;
                    }
                }
            }
            if( reverb_mode ) {
                while( 1 ) {
                    if( !SPI1STATbits.SPIRBE && !SPI1STATbits.SPITBF ) {
                        signed short val = SPI1BUF;
                        if( MuteSamples ) {
                            val = 0;
                            --MuteSamples;
                        }
                        DMA_Buffer[read_offset2] = ((signed short)DMA_Buffer[read_offset] * (1024 - mix_val) + (signed short)DMA_Buffer[read_offset2] * mix_val) >> 10;
                        val = ((signed short)DMA_Buffer[read_offset2] * (1024 - mix_val) + val * mix_val) >> 10;
                        DMA_Buffer[write_offset] = PORTDbits.RD7 ? val : 0;
                        if( ++write_offset == BUFFER_SIZE_WORDS )
                            write_offset = 0;
                        SPI1BUF = val;
                        if( ++read_offset == BUFFER_SIZE_WORDS )
                            read_offset = 0;
                        if( ++read_offset2 == BUFFER_SIZE_WORDS )
                            read_offset2 = 0;
                    } else {
                        break;
                    }
                }
            } else {
                while( 1 ) {
                    if( !SPI1STATbits.SPIRBE && !SPI1STATbits.SPITBF ) {
                        signed short val = SPI1BUF;
                        if( MuteSamples ) {
                            val = 0;
                            --MuteSamples;
                        }
                        val = ((signed short)DMA_Buffer[read_offset] * (1024 - mix_val) + val * mix_val) >> 10;
                        DMA_Buffer[write_offset] = PORTDbits.RD7 ? val : 0;
                        if( ++write_offset == BUFFER_SIZE_WORDS )
                            write_offset = 0;
                        SPI1BUF = val;
                        if( ++read_offset == BUFFER_SIZE_WORDS )
                            read_offset = 0;
                    } else {
                        break;
                    }
                }
            }

            if( !VR1_installed )
                new_delay = AN24Val;
            else
                new_delay = AN8Val;
            if( last_delay > 2 && new_delay <= last_delay - ADJUST_MIN_DELTA || new_delay >= last_delay + ADJUST_MIN_DELTA ) {
                // delay changed
                compute_delay(new_delay);

                read_offset = write_offset + (sizeof(DMA_Buffer) - internal_delay_bytes) / 2;
                if( read_offset > sizeof(DMA_Buffer) / 2 )
                    read_offset -= sizeof(DMA_Buffer) / 2;
                if( reverb_mode ) {
                    read_offset2 = write_offset + (sizeof(DMA_Buffer) - REVERB_SAMPLES) / 2;
//                    read_offset2 = write_offset + (sizeof(DMA_Buffer) - ((internal_delay_bytes * REVERB_FRACTION) & ~3)) / 2;
                    if( read_offset2 > sizeof(DMA_Buffer) / 2 )
                        read_offset2 -= sizeof(DMA_Buffer) / 2;
                }
                last_delay = new_delay;
            }

            if( VolumePot_installed )
                UpdateHeadphoneVol();
            if( !PORTCbits.RC14 != Mic_on )
                UpdateMicEnabled();
        }
    } else {
#ifndef USE_DMA
        write_offset = 0;
        read_offset = (sizeof(DMA_Buffer) - internal_delay_bytes) / 2;
#endif
        while(1) {
            if( last_delay < MIN_VAL_ZERO_DELAY ) {
                if( !passthrough_mode ) {
                    Bypass_on = true;
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    passthrough_mode = 1;
                }
            } else {
                if( passthrough_mode ) {
                    Bypass_on = false;
                    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
                    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
                    passthrough_mode = 0;
                }
            }

            if( external_delay_bytes ) {
                if( SRAM_readpos != SRAM_writepos ) {
#ifdef USE_DMA
                    current_DMA_write_offset = DCH0DPTR;
                    if( current_DMA_write_offset < copy_to_RAM_offset || current_DMA_write_offset >= copy_to_RAM_offset + SRAM_XFER_SIZE ) {
#else
                    if( write_offset < copy_to_RAM_offset || write_offset >= copy_to_RAM_offset + SRAM_XFER_SIZE ) {
#endif
                        RAMtoSRAMCopy((unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_writepos, SRAM_XFER_SIZE);
                        SRAM_writepos += SRAM_XFER_SIZE;
                        if( SRAM_writepos == SRAM_SIZE )
                            SRAM_writepos = 0;
                        SRAMtoRAMCopy(SRAM_readpos, (unsigned char*)DMA_Buffer + copy_to_RAM_offset, SRAM_XFER_SIZE);
                        SRAM_readpos += SRAM_XFER_SIZE;
                        if( SRAM_readpos == SRAM_SIZE )
                            SRAM_readpos = 0;
                        copy_to_RAM_offset += SRAM_XFER_SIZE;
                        if( copy_to_RAM_offset == sizeof(DMA_Buffer) )
                            copy_to_RAM_offset = 0;
                    }
                }
#ifdef USE_DMA
            } else {
                PowerSaveIdle();
            }
#else
            }
            while( 1 ) {
                if( !SPI1STATbits.SPIRBE ) {
                    signed short val = SPI1BUF;
                    if( MuteSamples ) {
                        val = 0;
                        --MuteSamples;
                    }
                    DMA_Buffer[write_offset] = val;
                    if( ++write_offset == BUFFER_SIZE_WORDS )
                        write_offset = 0;
                } else if( !SPI1STATbits.SPITBF ) {
                    SPI1BUF = DMA_Buffer[read_offset];
                    if( ++read_offset == BUFFER_SIZE_WORDS )
                        read_offset = 0;
                } else {
                    break;
                }
            }
#endif
            if( !VR1_installed )
                new_delay = AN24Val;
            else
                new_delay = AN8Val;
            if( last_delay > 2 && new_delay <= last_delay - ADJUST_MIN_DELTA || new_delay >= last_delay + ADJUST_MIN_DELTA ) {
                // delay changed
                compute_delay(new_delay);

#ifdef USE_DMA
                DCH1ECONbits.CABORT = 1; // abort/reset transmission
                DCH1CONbits.CHEN = 0; // stop transmission
                while( DCH0DPTR != internal_delay_bytes )
                    ;
                DCH1CONbits.CHEN = 1; // start transmission
#else
                read_offset = write_offset + (sizeof(DMA_Buffer) - internal_delay_bytes) / 2;
                if( read_offset > sizeof(DMA_Buffer) / 2 )
                    read_offset -= sizeof(DMA_Buffer) / 2;
#endif
                /*
                temp = DCH1SPTR;
                while( DCH1SPTR == temp || (DCH1SPTR&2) )
                    ;
                DCH1CONbits.CHEN = 0;   // pause transmission
                DCH1INTbits.CHSDIE = 1; // get interrupt when transmission completes
                DCH1INTbits.CHSDIF = 0; // get interrupt when transmission completes
                IEC2bits.DMA1IE = 1;    // enable DMA channel 1 interrupts
                IFS2bits.DMA1IF = 0;    // clear any existing DMA channel 1 interrupt flag
                IPC10bits.DMA1IP = 5;
                temp = internal_delay_bytes + sizeof(DMA_Buffer) - (DCH0SPTR & ~3);
                if( temp >= sizeof(DMA_Buffer) )
                    temp -= sizeof(DMA_Buffer);
                DCH1SSA = VirtToPhys(DMA_Buffer + temp / sizeof(*DMA_Buffer));// transfer source physical address
                DCH1SSIZ = sizeof(DMA_Buffer) - temp; // source size
                DCH1CONbits.CHAEN = 0;   // turn off auto-enable until interrupt resets it
                DCH1CONbits.CHEN = 1;    // resume transmission
*/
                last_delay = new_delay;
            }

            if( VolumePot_installed )
                UpdateHeadphoneVol();
            if( !PORTCbits.RC14 != Mic_on )
                UpdateMicEnabled();
        }
    }
#endif
}
